function [Z]=SPEI(P,PET,s,krnl)
%Standardized Precipitation Evapotranspiration Index
% [Z]=SPEI(P,PET,s)
%
% P = Vector of monthly precipitation values
% PET = Vector of monthly PET values corresponding to P
% s = Time s over which SPI is computed (e.g., 1,3,12,48 months)
% krnl = Kernel type for D sum
%
% Z = SPEI values for the given location
%
% Note that in order to keep the analysis to whole years, the remainder 
% months for the first year are set to NaNs, e.g., if s = 3 then the
% remaining 9 months of SPEI for that first year will not be estimated.
%
% Nathan Steiger, LDEO Apr 2016

nseas=12; % assume it's always monthly data
P=P(:);PET=PET(:);
D=P-PET;
erase_yr=ceil(s/12);

% D setting to scaled dataset
A1=zeros(length(D)-s+1,s);
for i=1:s; A1(:,i)=D(i:length(D)-s+i); end

if strcmp(krnl,'r')
    % Simple moving sum, i.e., rectangular kernel
    XS=sum(A1,2);
elseif strcmp(krnl,'g')
    % Gaussian kernel with max = 1 (not normalized area with max = 0.4); 
    % months further away from the primary are less important in the sum
    k = flip(1/(0.4*sqrt(2*pi)).*exp(-(0:(-3/(s-1)):-3).^2/2));
    A1k=bsxfun(@times,A1,k);
    XS=sum(A1k,2);
elseif strcmp(krnl,'e')
    % Decaying exponential 
    % months further away from the primary are less important in the sum
    k = flip(exp(0:(-3/(s-1)):-3));
    A1k=bsxfun(@times,A1,k);
    XS=sum(A1k,2);    
else
    error('Wrong kernel type...')
end


% Remove first portion of the record to have whole years
if(s>1); XS(1:nseas*erase_yr-s+1)=[]; end

Z=zeros(length(XS),1);
for i=1:nseas
    tind=i:nseas:length(XS);
    Xn=XS(tind);
    % Fit D to the generalized extreme value distribution using maximum
    % likelihood estimates of the parameters
    prm=gevfit(Xn);
    gev_D=gevcdf(Xn,prm(1),prm(2),prm(3));
    % Transform GEV CDF to normal CDF
    Z(tind)=norminv(gev_D);
end

% Add back removed portions of the record as NaNs
if (s > 1); Z=cat(1,NaN(erase_yr*nseas,1),Z); end
