function [X,xmeta] = load_S(mdl_c,state_tp,mon_avg_o,mon_avg_f,m_yrs)
%LOAD_S  Load climate model or reanalysis data for state vector
%   [X,xmeta] = load_S(model,state_tp,mon_avg_o,mon_avg_f,m_yrs)
%     Given a model code 'mdl_c', state vector variable type 'state_tp',
%     start and end months for averaging, and the years from the model
%     'm_yrs', this function will load the data saved to the computer
%     'humid'. The outputs are the state vector 'X', the latitudes and
%     longitudes of the gridded data, and the locations of the state
%     variables 'id_X' which are in the same order as given in 'state_tp'.
%     The state variables are annually averaged according to the start
%     month and end moth inputs, unless these are empty in which case the
%     original monthly data is returned. Also available for certain
%     variables are the ocean lats/lons: olat/olon
%
%     The possible data source codes are:
%     'ccsm4_lm'
%     'ccsm4_2mpi'
%     'ccsm4_2gfdlpic'
%     'ccsm4_pic'
%     'gfdlcm3_pic'
%     'gfdlcm3_hist'
%     'gfdlesm2g_pi'
%     'gfdlesm2m_pi'
%     'mpi_lm'
%     'echam5'
%     'echam5_2cam5iso'
%     'echam5_2speedy'
%     'speedy_lm'
%     'icam5_hist'
%     All full forcing simulations from 'cesm_lmeXXX'
%
%   Nathan Steiger, October 2015; Nov 2017

X=[];%id_X=zeros(2*length(state_tp),1);
xmeta=cell(0);

if strcmp(mdl_c(1:8),'cesm_lme')
    
    ens=mdl_c(9:11);
    varpth=['/d6/haibo/lme/cam5/BLMTRC5CN/ALLFORCEINGS/' ens];
    
    lat=ncread([varpth '/TREFHT/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.TREFHT.085001-184912.nc'],'lat');
    lon=ncread([varpth '/TREFHT/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.TREFHT.085001-184912.nc'],'lon');
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                % Surface temperature
                Xv_mon=ncread([varpth '/TREFHT/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.TREFHT.085001-184912.nc'],'TREFHT');
                Xv_mon=permute(Xv_mon,[2 1 3]);

% 		disp('Bias correcting surface temperature to climatology')
% 		[X_clim] = load_obsclim('bearth',lat,lon);
% 		[nt,nn,nm]=size(Xv_mon);
% 		Xv_mon12=reshape(Xv_mon,nt,nn,12,nm/12);
% 		Xv_ns=bsxfun(@minus,Xv_mon12,mean(Xv_mon12,4));% remove climatology
% 		Xv_bc=bsxfun(@plus,Xv_ns,X_clim); % replace climatology
% 		Xv_mon=reshape(Xv_bc,nt,nn,nm);	

		Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                
               	lat=ncread([varpth '/TREFHT/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.TREFHT.085001-184912.nc'],'lat');
    		lon=ncread([varpth '/TREFHT/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.TREFHT.085001-184912.nc'],'lon');
		disp('!!! Coarsened 2m T for PAGES reconstruction !!!')
		hlatlon=load('/d1/nsteiger/climate-data/hadcrut/HadCRUT4.3_GraphEM_SP80_18502014.mat','lat','lon');
		clat=hlatlon.lat;clon=hlatlon.lon;
		Xv=coarser(lat,lon,Xv,clat,clon);
		lat=clat;lon=clon;

		[rows,cols,len]=size(Xv(:,:,m_yrs));
                id_x0=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='2 m temperature';
		xmeta{j}.lat=lat;
		xmeta{j}.lon=lon;
		xmeta{j}.id_X=[id_x0,id_x1];
            case '2'
                % Global mean 2m temperature
                Xv_mon=ncread([varpth '/TREFHT/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.TREFHT.085001-184912.nc'],'TREFHT');
                Xv_mon=permute(Xv_mon,[2 1 3]);

		disp('GMT: Bias correcting surface temperature to climatology')
		[X_clim] = load_obsclim('bearth',lat,lon);
		[nt,nn,nm]=size(Xv_mon);
		Xv_mon12=reshape(Xv_mon,nt,nn,12,nm/12);
		Xv_ns=bsxfun(@minus,Xv_mon12,mean(Xv_mon12,4));% remove climatology
		Xv_bc=bsxfun(@plus,Xv_ns,X_clim); % replace climatology
		Xv_mon=reshape(Xv_bc,nt,nn,nm);	

		Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                
		%Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                %Xv=permute(Xv,[2 1 3]);
		
    		%lat=ncread([varpth '/TREFHT/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.TREFHT.085001-184912.nc'],'lat');
    		%lon=ncread([varpth '/TREFHT/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.TREFHT.085001-184912.nc'],'lon');
		%disp('!!! Coarsened 2m T for PAGES reconstruction !!!')
		%hlatlon=load('/d1/nsteiger/climate-data/hadcrut/HadCRUT4.3_GraphEM_SP80_18502014.mat','lat','lon');
		%clat=hlatlon.lat;clon=hlatlon.lon;
		%Xv=coarser(lat,lon,Xv,clat,clon);
		%lat=clat;lon=clon;
                
		gmt=wmean(Xv,lat);
                len=length(gmt(m_yrs));
                %id_X(2*j-1)=size(X,1)+1;
                %X=cat(1,X,reshape(gmt(m_yrs),1,len));
                %id_X(2*j)=size(X,1);
                %clear Xv Xv_mon
                id_x0=size(X,1)+1;
                X=cat(1,X,reshape(gmt(m_yrs),1,len));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='global mean 2 m temperature';
		xmeta{j}.id_X=[id_x0,id_x1];
            case 's'
                % SST
                olat=permute(ncread(['/d1/nsteiger/climate-data/cesm-lme/b.e11.BLMTRC5CN.f19_g16.' ens '.pop.h.SST.085001-184912.nc'],'TLAT'),[2 1]);
                olon=permute(ncread(['/d1/nsteiger/climate-data/cesm-lme/b.e11.BLMTRC5CN.f19_g16.' ens '.pop.h.SST.085001-184912.nc'],'TLONG'),[2 1]);
                Xv_mon=squeeze(ncread(['/d1/nsteiger/climate-data/cesm-lme/b.e11.BLMTRC5CN.f19_g16.' ens '.pop.h.SST.085001-184912.nc'],'SST'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                Xv(Xv==0)=NaN; % replace nans
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_x0=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='sea surface temperature';
		xmeta{j}.olat=olat;
		xmeta{j}.olon=olon;
		xmeta{j}.id_X=[id_x0,id_x1];
            case 'h'
                % N_HEAT
                auxlat=ncread(['/d1/nsteiger/climate-data/cesm-lme/b.e11.BLMTRC5CN.f19_g16.' ens '.pop.h.N_HEAT.085001-184912.nc'],'lat_aux_grid');
                Xv_mon=squeeze(ncread(['/d1/nsteiger/climate-data/cesm-lme/b.e11.BLMTRC5CN.f19_g16.' ens '.pop.h.N_HEAT.085001-184912.nc'],'N_HEAT'));
                Xv=mon2ann(squeeze(Xv_mon(:,1,1,:)),mon_avg_o,mon_avg_f);
                %Xv(Xv==0)=NaN; % replace nans => there are nans near
                %southernmost boundary, not sure what to do with those
                id_x0=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='northward heat transport';
		xmeta{j}.id_X=[id_x0,id_x1];
            case 'y'
 		% Sea surface salinit(y)
                olat=permute(ncread(['/d1/nsteiger/climate-data/cesm-lme/b.e11.BLMTRC5CN.f19_g16.' ens '.pop.h.SSS.085001-184912.nc'],'TLAT'),[2 1]);
                olon=permute(ncread(['/d1/nsteiger/climate-data/cesm-lme/b.e11.BLMTRC5CN.f19_g16.' ens '.pop.h.SSS.085001-184912.nc'],'TLONG'),[2 1]);
                Xv_mon=squeeze(ncread(['/d1/nsteiger/climate-data/cesm-lme/b.e11.BLMTRC5CN.f19_g16.' ens '.pop.h.SSS.085001-184912.nc'],'SSS'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                Xv(Xv==0)=NaN; % replace nans
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_x0=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='sea surface salinity';
		xmeta{j}.olat=olat;
		xmeta{j}.olon=olon;
		xmeta{j}.id_X=[id_x0,id_x1];
            case 'p'
                % Precipitation
                %Xv_mon1=ncread([varpth '/RAIN/b.e11.BLMTRC5CN.f19_g16.' ens '.clm2.h0.RAIN.085001-184912.nc'],'RAIN');
                %Xv_mon2=ncread([varpth '/SNOW/b.e11.BLMTRC5CN.f19_g16.' ens '.clm2.h0.SNOW.085001-184912.nc'],'SNOW');
                %Xv_mon=Xv_mon1+Xv_mon2; % precip = rain + snow
                Xv_mon1=ncread(['/d1/nsteiger/climate-data/cesm-lme/pr/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.PRECC.085001-184912.nc'],'PRECC');
                Xv_mon2=ncread(['/d1/nsteiger/climate-data/cesm-lme/pr/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.PRECL.085001-184912.nc'],'PRECL');
                Xv_mon=Xv_mon1+Xv_mon2; % precip = convective + large scale

                Xv_mon=permute(Xv_mon,[2 1 3]);

		disp('Precip: Bias correcting precipitation to climatology')
		[X_clim] = load_obsclim('gpcp',lat,lon);
		[nt,nn,nm]=size(Xv_mon);
		% Convert precipitation from m/s to mm/day (1 day = 86400 sec)
		Xv_mon=Xv_mon*86400*1000;
		Xv_mon12=reshape(Xv_mon,nt,nn,12,nm/12);
		Xv_ns=bsxfun(@minus,Xv_mon12,mean(Xv_mon12,4));% remove climatology
		Xv_bc=bsxfun(@plus,Xv_ns,X_clim); % replace climatology
		% Replace zero or less with gamma random numbers with a mean of 0.05, a value much less than error in clim obs.
		%  In distribution plots, this gamma error makes the distributions at small values look very much like
		%  the original model distributions (when it has been converted into the appropriate units)
		smlidx=find(Xv_bc<=0);Xv_bc(smlidx)=gamrnd(1,0.05,length(smlidx),1);
		Xv_mon=reshape(Xv_bc,nt,nn,nm);	

		Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);


                %Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                %Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_x0=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='precipitation';
		xmeta{j}.lat=lat;
		xmeta{j}.lon=lon;
		xmeta{j}.id_X=[id_x0,id_x1];
            case 'g'
                % Geopotential
                Xv_mon=ncread(['/d1/nsteiger/climate-data/cesm-lme/zg-850-500-200_Amon_CESMLME_' ens '_085001-184912.nc'],'Z');
                %Xv_mon=squeeze(Xv_mon(:,:,2,:)); % 500 mb
                Xv_mon=squeeze(Xv_mon(:,:,3,:)); % 200 mb
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_x0=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='geopotential height';
		xmeta{j}.lat=lat;
		xmeta{j}.lon=lon;
		xmeta{j}.id_X=[id_x0,id_x1];
            case 'l'
                % Soil moisture
                %load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_mrso_3m_landonly.mat'])
                load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_mrso_1m_landonly.mat']);disp('1 m soil moisture')
                %load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_mrso_37cm_landonly.mat']);disp('37 cm soil moisture')
                %load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_mrso_11cm_landonly.mat']);disp('11 cm soil moisture')
                %mon_o=12;mon_f=2; disp('DJF soil moist')
                %mon_o=6;mon_f=8; disp('JJA soil moist')
                %Xv=mon2ann(mrso_mon,mon_o,mon_f);
                %if(mon_avg_o>mon_avg_f)&&(mon_f>mon_o);Xv=Xv(:,1:end-1);end
                Xv=mon2ann(mrso_mon,mon_avg_o,mon_avg_f);
                id_x0=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='soil moisture';
		xmeta{j}.lat=lat;
		xmeta{j}.lon=lon;
		xmeta{j}.id_X=[id_x0,id_x1];
                
            case 'i'
                % SPEI 12
                %load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_spei_output_scl_12_krnl_e_u2_gcm_09-Jun-2016_10:30:26.mat'],'spei_lndonly')
                %mon_o=12;mon_f=2; disp('DJF SPEI12')
                %mon_o=6;mon_f=8; disp('JJA SPEI12')
                %Xv=mon2ann(spei_lndonly,mon_o,mon_f);
                %if(mon_avg_o>mon_avg_f)&&(mon_f>mon_o);Xv=Xv(:,1:end-1);end
                %Xv=mon2ann(spei_lndonly,mon_avg_o,mon_avg_f);
                %id_X(2*j-1)=size(X,1)+1;
                %X=cat(1,X,Xv(:,m_yrs));
                %id_X(2*j)=size(X,1);
                %clear Xv
                
                disp(' !! Loading full, bias-corrected, SPEI field (NaNs included) !! ')
                load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_spei_output_biascorr_scl_12_krnl_e_u2_gcm.mat'],'spei_f')
                Xv=mon2ann(spei_f,mon_avg_o,mon_avg_f);
                Xv(Xv==0)=NaN;
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_x0=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='SPEI 12 month';
		xmeta{j}.lat=lat;
		xmeta{j}.lon=lon;
		xmeta{j}.id_X=[id_x0,id_x1];
                
            case 'r'
                % PDSI (palme*r* drought severity index)
                % single uniform available water content
                %load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_pdsi_output_AWC_c_7_u2_gcm_09-Jun-2016_10:45:14.mat'],'pdsi_lndonly')
                %mon_o=12;mon_f=2; disp('DJF PDSI')
                %mon_o=6;mon_f=8; disp('JJA PDSI')
                %Xv=mon2ann(pdsi_lndonly,mon_o,mon_f);
                %if(mon_avg_o>mon_avg_f)&&(mon_f>mon_o);Xv=Xv(:,1:end-1);end
                %Xv=mon2ann(pdsi_lndonly,mon_avg_o,mon_avg_f);
                %id_X(2*j-1)=size(X,1)+1;
                %X=cat(1,X,Xv(:,:,m_yrs));
                %id_X(2*j)=size(X,1);
                %clear Xv
                
                disp(' !! Loading full, bias-corrected, PDSI field (NaNs included) !! ')
		load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_pdsi_output_biascorr_AWC_c_7_u2_gcm.mat'],'pdsi_f')             
   		%load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_pdsi_output_AWC_c_7_u2_gcm.mat'],'pdsi_f')
                Xv=mon2ann(pdsi_f,mon_avg_o,mon_avg_f);
                Xv(Xv==0)=NaN;
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_x0=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='PDSI';
		xmeta{j}.lat=lat;
		xmeta{j}.lon=lon;
		xmeta{j}.id_X=[id_x0,id_x1];
                
            case 'j'
                % SPEI 12 (land only -- for state vector...)
                
                %load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_spei_output_scl_12_krnl_e_u2_gcm.mat'],'spei_lndonly')
                disp('Bias-corrected SPEI')
		load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_spei_output_biascorr_scl_12_krnl_e_u2_gcm.mat'],'spei_lndonly','lndidx')
 		Xv=mon2ann(spei_lndonly,mon_avg_o,mon_avg_f);
                Xv(Xv==0)=NaN;
                id_x0=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='SPEI 12 months (land only)';
		xmeta{j}.lat=lat;
		xmeta{j}.lon=lon;
		xmeta{j}.id_X=[id_x0,id_x1];
		xmeta{j}.lndidx=lndidx;
                
            case 'q'
                % PDSI (land-only--for state vector...)
                % single uniform available water content
                
                %load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_pdsi_output_AWC_c_7_u2_gcm.mat'],'pdsi_lndonly')
                disp('Bias-corrected PDSI')
		load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_pdsi_output_biascorr_AWC_c_7_u2_gcm.mat'],'pdsi_lndonly','lndidx')   
                Xv=mon2ann(pdsi_lndonly,mon_avg_o,mon_avg_f);
                Xv(Xv==0)=NaN;
                id_x0=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='PDSI (land only)';
		xmeta{j}.lat=lat;
		xmeta{j}.lon=lon;
		xmeta{j}.id_X=[id_x0,id_x1];
		xmeta{j}.lndidx=lndidx;
                
            case 'w'
                % PDSI (palme*r* drought severity index) in North
                % American southwest (index time series for tests)
                % single uniform available water content
                %load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_pdsi_output_AWC_c_7_u2_gcm.mat'],'pdsi_f')
		load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_pdsi_output_biascorr_AWC_c_7_u2_gcm.mat'],'pdsi_f')             
                %mon_o=12;mon_f=2; disp('DJF PDSI')
                mon_o=6;mon_f=8; disp('JJA PDSI')
                %mon_o=5;mon_f=9; disp('MJJAS PDSI')
                Xv=mon2ann(pdsi_f,mon_o,mon_f);
                %if(mon_avg_o>mon_avg_f)&&(mon_f>mon_o);Xv=Xv(:,1:end-1);end
                
                %Xv=mon2ann(pdsi_f,mon_avg_o,mon_avg_f);
                % COMPUTE SINGLE INDEX
                load('/d1/nsteiger/climate-data/cesm-lme/Nrth_A_Swthwst_mask.mat')
                nasw(nasw==0)=NaN;
                Xv_msk=bsxfun(@times,Xv,nasw);
                A=cosd(repmat(lat,[1 length(lon)]));
                % Compute the index and add it to state vector
                Di=wmean_a(Xv_msk,A);
                len=length(Di(m_yrs));
                id_x0=size(X,1)+1;
                X=cat(1,X,reshape(Di(m_yrs),1,len));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='PDSI NASW';
		xmeta{j}.lat=lat;
		xmeta{j}.lon=lon;
		xmeta{j}.id_X=[id_x0,id_x1];
                
            case 'e'
                % PDSI (palme*r* drought severity index) in East Africa
                % single uniform available water content
                %load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_spei_output_scl_12_krnl_e_u2_gcm.mat'],'spei_f')
                %Xv=mon2ann(spei_f,mon_avg_o,mon_avg_f);
                
		load(['/d1/nsteiger/climate-data/cesm-lme/cesm_lme_' ens '_pdsi_output_AWC_c_7_u2_gcm.mat'],'pdsi_f')
                %mon_o=12;mon_f=2; disp('DJF PDSI')
                %mon_o=6;mon_f=8; disp('JJA PDSI')
                %Xv=mon2ann(pdsi_f,mon_o,mon_f);
                %if(mon_avg_o>mon_avg_f)&&(mon_f>mon_o);Xv=Xv(:,1:end-1);end
                
                Xv=mon2ann(pdsi_f,mon_avg_o,mon_avg_f);
                % COMPUTE SINGLE INDEX
                load('/d1/nsteiger/climate-data/cesm-lme/Est_Afrca_mask.mat')
                nasw=estafrca;% just use previous code
                nasw(nasw==0)=NaN;
                Xv_msk=bsxfun(@times,Xv,nasw);
                A=cosd(repmat(lat,[1 length(lon)]));
                % Compute the index and add it to state vector
                Di=wmean_a(Xv_msk,A);
                len=length(Di(m_yrs));

                id_x0=size(X,1)+1;
                X=cat(1,X,reshape(Di(m_yrs),1,len));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='PDSI East Africa';
		xmeta{j}.lat=lat;
		xmeta{j}.lon=lon;
		xmeta{j}.id_X=[id_x0,id_x1];

            case 'o'
                % AMO (annual, using skin temperature on regular lat/lon grid)
                Xv_mon=ncread([varpth '/TS/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.TS.085001-184912.nc'],'TS');
               	
                Xv_mon=permute(Xv_mon,[2 1 3]);

		disp('AMO: Bias correcting surface temperature to climatology')
		[X_clim] = load_obsclim('bearth',lat,lon);
		[nt,nn,nm]=size(Xv_mon);
		Xv_mon12=reshape(Xv_mon,nt,nn,12,nm/12);
		Xv_ns=bsxfun(@minus,Xv_mon12,mean(Xv_mon12,4));% remove climatology
		Xv_bc=bsxfun(@plus,Xv_ns,X_clim); % replace climatology
		Xv_mon=reshape(Xv_bc,nt,nn,nm);	

		Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);

		%Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                %Xv=permute(Xv,[2 1 3]);
                load('/d1/nsteiger/climate-data/cesm-lme/north_atlantic_ocean_mask.mat')
                AMO = amo(Xv,atlntc_mask,lat,0);%0=no smoothing
                len=length(AMO(m_yrs));
                id_x0=size(X,1)+1;
                X=cat(1,X,reshape(AMO(m_yrs),1,len));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
		xmeta{j}.varnmlng='AMO (NASST index)';
		xmeta{j}.id_X=[id_x0,id_x1];

            case 'n'
                % Nino indices (monthly, using skin temperature on regular lat/lon grid)
                Xv_mon=ncread([varpth '/TS/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.TS.085001-184912.nc'],'TS');
                Xv_mon=permute(Xv_mon,[2 1 3]);
       
		disp('Nino indices: Bias correcting surface temperature to climatology')
		[X_clim] = load_obsclim('bearth',lat,lon);
		[nt,nn,nm]=size(Xv_mon);
		Xv_mon12=reshape(Xv_mon,nt,nn,12,nm/12);
		Xv_ns=bsxfun(@minus,Xv_mon12,mean(Xv_mon12,4));% remove climatology
		Xv_bc=bsxfun(@plus,Xv_ns,X_clim); % replace climatology
		Xv_mon=reshape(Xv_bc,nt,nn,nm);	

                % Compute all the nino indices
                %bc='y';disp('Bias-correcting Nino index');% option corrects index; field is already corrected above
                bc='n';
		nr=[];nmeta=cell(0);
                for ni=1:5
		   [nv,nmeta0]=nino(Xv_mon,lat,lon,ni,bc);
		   % Account for different year definitions
		   len=size(Xv_mon,3);
		   if mon_avg_o==4 && mon_avg_f==3
		      nv=nv(mon_avg_o:(len-(12-mon_avg_f)));
		   elseif mon_avg_o==1 && mon_avg_f==12
		      nv=nv(mon_avg_o:(len-(12-mon_avg_f)));
		   else 
		      disp('Just using tropical year for monthly Nino index for seasonal recons')
		      nv=nv(4:(len-(12-3)));
		   end
		   % Reshape index so that each year has 12 monthly values
		   n0=reshape(nv,12,length(nv)/12);
		   nr=cat(1,nr,n0);
		   nmeta{ni}=nmeta0;
                end   

                id_x0=size(X,1)+1;
                X=cat(1,X,nr(:,m_yrs));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
	        xmeta{j}.indinfo=nmeta; 	
		xmeta{j}.varnmlng='Nino indices';
		xmeta{j}.id_X=[id_x0,id_x1];


            case 'z'
                % ITCZ location (can be seasonal or annual)
                Xv_mon1=ncread(['/d1/nsteiger/climate-data/cesm-lme/pr/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.PRECC.085001-184912.nc'],'PRECC');
                Xv_mon2=ncread(['/d1/nsteiger/climate-data/cesm-lme/pr/b.e11.BLMTRC5CN.f19_g16.' ens '.cam.h0.PRECL.085001-184912.nc'],'PRECL');
                Xv_mon=Xv_mon1+Xv_mon2; % precip = convective + large scale
                Xv_mon=permute(Xv_mon,[2 1 3]);

		disp('ITCZ: Bias correcting precipitation to climatology')
		[X_clim] = load_obsclim('gpcp',lat,lon);
		[nt,nn,nm]=size(Xv_mon);
		% Convert precipitation from m/s to mm/day (1 day = 86400 sec)
		Xv_mon=Xv_mon*86400*1000;
		Xv_mon12=reshape(Xv_mon,nt,nn,12,nm/12);
		Xv_ns=bsxfun(@minus,Xv_mon12,mean(Xv_mon12,4));% remove climatology
		Xv_bc=bsxfun(@plus,Xv_ns,X_clim); % replace climatology
		% Replace zero or less with gamma random numbers with a mean of 0.05, a value much less than error in clim obs.
		%  In distribution plots, this gamma error makes the distributions at small values look very much like
		%  the original model distributions (when it has been converted into the appropriate units)
		smlidx=find(Xv_bc<=0);Xv_bc(smlidx)=gamrnd(1,0.05,length(smlidx),1);
		Xv_mon=reshape(Xv_bc,nt,nn,nm);	

		Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                
		%Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                %Xv=permute(Xv,[2 1 3]);
                
		%disp('Just ITCZ over East Africa...')
                %ITCZ = itcz_lon(Xv,lat,lon,[28 52]);
                
		% ITCZ location over the ocean regions
		
		ilons={[320 345],[130 260],[160 260],[130 170],[170 260],[65 95],[50 95],[260 320],[95 130],[345 50],[28 50]};
		ilnf={'Atlantic','Pacific','Pacific (Schneider 2014)','East Pacific','West Pacific','South Asian Monsoon (Schneider 2014)',...
			'Indian','South America','Indonesia','Africa','East Africa'};
		imeta=cell(0);ITCZ=[];
		for ii=1:length(ilons);
		   [itcz_mx,~,imeta0] = itcz_lon(Xv,lat,lon,ilons{ii});                   		
		   imeta{ii}=imeta0;
		   imeta{ii}.varnmlng=ilnf{ii};
		   ITCZ=cat(1,ITCZ,itcz_mx);
		end

		%Atlantic = [311 350] my original values
		%ITCZ(1,:) = itcz_lon(Xv,lat,lon,[320 345]);	
		%Pacific = [128 278] my original values
		%ITCZ(2,:) = itcz_lon(Xv,lat,lon,[130 260]);% full pacific
		%ITCZ(3,:) = itcz_lon(Xv,lat,lon,[160 260]);% full pacific (Schneider 2014)
		%ITCZ(4,:) = itcz_lon(Xv,lat,lon,[130 170]);% east pacific
		%ITCZ(5,:) = itcz_lon(Xv,lat,lon,[170 260]);% west pacific
		%Indian = [50 96] my original values
		%ITCZ(6,:) = itcz_lon(Xv,lat,lon,[65 95]); % south asian monsoon (Schneider 2014)
		%ITCZ(7,:) = itcz_lon(Xv,lat,lon,[50 95]);
                % ITCZ over land regions
		%ITCZ(8,:) = itcz_lon(Xv,lat,lon,[260 320]); % south america
		%ITCZ(9,:) = itcz_lon(Xv,lat,lon,[95 130]); % indonesia
		%ITCZ(10,:) = itcz_lon(Xv,lat,lon,[345 50]); % africa
		%ITCZ(11,:) = itcz_lon(Xv,lat,lon,[28 50]); % east africa

                id_x0=size(X,1)+1;
                X=cat(1,X,ITCZ(:,m_yrs));
                id_x1=size(X,1);
                clear Xv Xv_mon
		xmeta{j}.varnm=state_tp(j);
	        xmeta{j}.indinfo=imeta; 	
		xmeta{j}.varnmlng='ITCZ indices';
		xmeta{j}.id_X=[id_x0,id_x1];

            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'ccsm4_lm')
    
    load('/d1/nsteiger/climate-data/ccsm4_last_mil/latlon.mat')
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                
                % Surface temperature
                load('/d1/nsteiger/climate-data/ccsm4_last_mil/tas_past1000.mat')
                Xv=mon2ann(tas,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear tas
            case 'g'
                % Geopotential
                load('/d1/nsteiger/climate-data/ccsm4_last_mil/zg500_ccsm4_lm.mat')
                Xv=mon2ann(zg500_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear zg500_mon
            case 'u'
                % U200 mb winds (subtropical jet)
                load('/d1/nsteiger/climate-data/ccsm4_last_mil/ua200mb_mon.mat')
                Xv=mon2ann(ua_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear ua_mon
            case 'p'
                % Precipitation [kg m-2 s-1]
                load('/d1/nsteiger/climate-data/ccsm4_last_mil/pr_mon_ccsm4_past1000.mat')
                Xv=mon2ann(pr_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear pr_mon
            case 'e'
                % Precipitation minus Evaporation [kg m-2 s-1]
                load('/d1/nsteiger/climate-data/ccsm4_last_mil/p_e_ann_ccsm4_past1000.mat')
                Xv=mon2ann(pr_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear pr_mon
            case 'i'
                % ITCZ latitude
                load('/d1/nsteiger/climate-data/ccsm4_last_mil/itcz_lat_mon.mat')
                Xv=mon2ann(itcz_lat,mon_avg_o,mon_avg_f);
                len=length(Xv(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(m_yrs),1,len));
                id_X(2*j)=size(X,1);
            case 'a'
                % MOC index: atlantic or global
                load('/d1/nsteiger/climate-data/ccsm4_last_mil/amoc_ccsm4_lm.mat')
                amoc_i=mon2ann(amoc_mon,mon_avg_o,mon_avg_f);
                %load('/d1/nsteiger/climate-data/ccsm4_last_mil/gmoc_index_past1000.mat')
                %amoc_i=mon2ann(gmoc_mon,mon_avg_o,mon_avg_f);
                len=length(amoc_i(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(amoc_i(m_yrs),1,len));
                id_X(2*j)=size(X,1);
            case 'm'
                % Full MOC variable: atlantic or global
                load('/d1/nsteiger/climate-data/ccsm4_last_mil/atlmoc_past1000.mat')
                atlmoc=mon2ann(atlmoc,mon_avg_o,mon_avg_f);
                %load('/d1/nsteiger/climate-data/ccsm4_last_mil/gmoc_past1000.mat')
                %atlmoc=mon2ann(gmoc,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(atlmoc(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(atlmoc(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear atlmoc
            case 'o'
                % AMO index (Trenberth definition)
                load('/d1/nsteiger/climate-data/ccsm4_last_mil/amo_mon_anom_trenberth.mat')
                amo_i=mon2ann(amo_mon,mon_avg_o,mon_avg_f);
                len=length(amo_i(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(amo_i(m_yrs),1,len));
                id_X(2*j)=size(X,1);
            case 'h'
                % Ocean heat content (global or atlantic)
                %load('/d1/nsteiger/climate-data/ccsm4_last_mil/ohc_ann_atl_700m.mat')
                load('/d1/nsteiger/climate-data/ccsm4_last_mil/ohc_ann_glbl_700m.mat')
                ohc_i=ohc_glbl_m; % already annual average
                len=length(ohc_i(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(ohc_i(m_yrs),1,len));
                id_X(2*j)=size(X,1);
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'ccsm4_2mpi')
    
    % COARSENED VERSION OF CCSM4 TO MATCH MPI
    load('/d1/nsteiger/climate-data/mpi_esm_last_mil/latlon_mpi.mat')
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                
                % Surface temperature
                load('/d1/nsteiger/climate-data/ccsm4_last_mil/tas_mon_past1000_coarsenedMPIESM.mat')
                Xv=mon2ann(tas,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear tas
                
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'ccsm4_2gfdlpic')
    
    % COARSENED VERSION OF CCSM4 TO MATCH GFDL PI CONTROL
    load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/gfdl_latlon.mat')
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                
                % Surface temperature
                load('/d1/nsteiger/climate-data/ccsm4-pi-control/tas_mon_ccsm4_pic_coarsened2GFDL.mat')
                Xv=mon2ann(tas_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear tas
                
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'ccsm4_pic')
    
    load('/d1/nsteiger/climate-data/ccsm4-pi-control/latlon_ccsm4pi.mat')
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                
                % Surface temperature
                load('/d1/nsteiger/climate-data/ccsm4-pi-control/tas_mon_ccsm4_pic.mat')
                Xv=mon2ann(tas_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear tas
            case 'a'
                
                % MOC index: atlantic or global
                load('/d1/nsteiger/climate-data/ccsm4-pi-control/amocidx_mon_ccsm4_pic.mat')
                amoc_i=mon2ann(amocidx_mon,mon_avg_o,mon_avg_f);
                len=length(amoc_i(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(amoc_i(m_yrs),1,len));
                id_X(2*j)=size(X,1);
            case 'h'
                % Ocean heat content (global or atlantic)
                load('/d1/nsteiger/climate-data/ccsm4-pi-control/ohc_ann_glbl_700m.mat')
                ohc_i=ohc_glbl_m; % already annual average
                len=length(ohc_i(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(ohc_i(m_yrs),1,len));
                id_X(2*j)=size(X,1);
                
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'gfdlcm3_pic')
    
    load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/gfdl_latlon.mat')
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                % Surface temperature
                load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/tas_mon_gfdl_800.mat')
                %load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/tas_mon_gfdl_500.mat')
                Xv=mon2ann(tas_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear tas_mon
            case 'u'
                % U200 mb winds (subtropical jet)
                load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/ua200_mon.mat')
                Xv=mon2ann(ua200,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear ua200
            case 'g'
                % Geopotential
                load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/zg500_mon.mat')
                Xv=mon2ann(zg500,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear zg500
            case 'p'
                % Precipitation minus Evaporation [kg m-2 s-1]
                load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/p_e_mon_pic.mat')
                Xv=mon2ann(p_e,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
            case 'i'
                % ITCZ latitude
                load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/itcz_lat_mon.mat')
                Xv=mon2ann(itcz_lat,mon_avg_o,mon_avg_f);
                len=length(Xv(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(m_yrs),1,len));
                id_X(2*j)=size(X,1);
            case 'a'
                % MOC index: atlantic or global
                load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/amocidx_mon_gfdlcm3_pic.mat')
                amoc_i=mon2ann(amocidx_mon,mon_avg_o,mon_avg_f);
                %load('/d1/nsteiger/climate-data/ccsm4_last_mil/gmoc_index_past1000.mat')
                %amoc_i=mon2ann(gmoc_mon,mon_avg_o,mon_avg_f);
                len=length(amoc_i(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(amoc_i(m_yrs),1,len));
                id_X(2*j)=size(X,1);
            case 'm'
                % Full MOC variable: atlantic or global
                load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/atlmoc_mon_gfdl_800.mat')
                atlmoc=mon2ann(atlmoc,mon_avg_o,mon_avg_f);
                %load('/d1/nsteiger/climate-data/ccsm4_last_mil/gmoc_past1000.mat')
                %atlmoc=mon2ann(atlmoc,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(atlmoc(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(atlmoc(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear atlmoc
            case 'o'
                % AMO index (Trenberth definition)
                load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/amo_mon_anom_trenberth.mat')
                amo_i=mon2ann(amo_mon,mon_avg_o,mon_avg_f);
                len=length(amo_i(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(amo_i(m_yrs),1,len));
                id_X(2*j)=size(X,1);
            case 'h'
                % Ocean heat content (global or atlantic)
                %load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/ohc_ann_atl_700m.mat')
                load('/d1/nsteiger/climate-data/gfdl-cm3-pi-control/ohc_ann_glbl_700m.mat')
                ohc_i=ohc_glbl_m; % already annual average
                len=length(ohc_i(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(ohc_i(m_yrs),1,len));
                id_X(2*j)=size(X,1);
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'gfdlcm3_hist')
    
    lat=ncread('/d1/nsteiger/climate-data/gfdl-cm3-hist/tas_Amon_GFDL-CM3_historical_r1i1p1_186001-200512.nc','lat');
    lon=ncread('/d1/nsteiger/climate-data/gfdl-cm3-hist/tas_Amon_GFDL-CM3_historical_r1i1p1_186001-200512.nc','lon');
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                % Surface temperature
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-cm3-hist/tas_Amon_GFDL-CM3_historical_r1i1p1_186001-200512.nc','tas');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'g'
                % Geopotential
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/gfdl-cm3-hist/zg500_Amon_GFDL-CM3_historical_r1i1p1_186001-200512.nc','zg'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'p'
                % Precipitation
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/gfdl-cm3-hist/pr_Amon_GFDL-CM3_historical_r1i1p1_186001-200512.nc','pr'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'l'
                % Soil moisture
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/gfdl-cm3-hist/mrsos_Lmon_GFDL-CM3_historical_r1i1p1_186001-200512.nc','mrsos'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'w'
                % surface wind
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/gfdl-cm3-hist/sfcWind_Amon_GFDL-CM3_historical_r1i1p1_186001-200512.nc','sfcWind'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 's'
                % sea level pressure
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/gfdl-cm3-hist/psl_Amon_GFDL-CM3_historical_r1i1p1_186001-200512.nc','psl'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'gfdlesm2g_pi')
    
    lat=ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/tas_Amon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','lat');
    lon=ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/tas_Amon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','lon');
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                % Surface temperature
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/tas_Amon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','tas');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'g'
                % Geopotential height
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/zg200_Amon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','zg'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'l'
                % Soil moisture
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/mrso_Lmon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','mrso');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                % Just use land area
                sftlf=permute(ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/sftlf_fx_GFDL-ESM2M_piControl_r0i0p0.nc','sftlf'),[2 1]);
                lndidx=find((sftlf(:)>0));
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                Xv=reshape(Xv(:,:,m_yrs),rows*cols,len);
                Xv=Xv(lndidx,:);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,Xv);
                id_X(2*j)=size(X,1);
            case 'z'
                % ITCZ location (can be seasonal or annual)
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/pr_Amon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','pr');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                ITCZ = itcz(Xv,lat,lon);
                len=length(ITCZ(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(ITCZ(m_yrs),1,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'i'
                % SPEI 12
                load('/d1/nsteiger/climate-data/gfdl-esm2g-pi/spei_output_scl_12_krnl_e_u2_gcm_04-Aug-2016_11:28:54.mat','spei_lndonly')
                %mon_o=12;mon_f=2; disp('DJF SPEI12')
                %mon_o=6;mon_f=8; disp('JJA SPEI12')
                %Xv=mon2ann(spei_lndonly,mon_o,mon_f);
                %if(mon_avg_o>mon_avg_f)&&(mon_f>mon_o);Xv=Xv(:,1:end-1);end
                Xv=mon2ann(spei_lndonly,mon_avg_o,mon_avg_f);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_X(2*j)=size(X,1);
                clear Xv
                
            case 'r'
                % PDSI (palme*r* drought severity index)
                % single uniform available water content
                load('/d1/nsteiger/climate-data/gfdl-esm2g-pi/pdsi_output_AWC_c_7_u2_gcm_04-Aug-2016_11:31:32.mat','pdsi_lndonly')
                %mon_o=12;mon_f=2; disp('DJF PDSI')
                %mon_o=6;mon_f=8; disp('JJA PDSI')
                %Xv=mon2ann(pdsi_lndonly,mon_o,mon_f);
                %if(mon_avg_o>mon_avg_f)&&(mon_f>mon_o);Xv=Xv(:,1:end-1);end
                Xv=mon2ann(pdsi_lndonly,mon_avg_o,mon_avg_f);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_X(2*j)=size(X,1);
                clear Xv
            case 'o'
                % AMO (annual, using skin temperature on regular lat/lon grid)
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/ts_Amon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','ts');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                load('/d1/nsteiger/climate-data/gfdl-esm2g-pi/north_atlantic_ocean_mask.mat')
                AMO = amo(Xv,atlntc_mask,lat,0);%0=no smoothing
                len=length(AMO(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(AMO(m_yrs),1,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'n'
                % Nino3.4 (monthly, using skin temperature on regular lat/lon grid)
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/ts_Amon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','ts');
                Xv_mon=permute(Xv_mon,[2 1 3]);
                % compute nino index
                nv=nino(Xv_mon,lat,lon,3);
                % Account for different year definitions
                len=size(Xv_mon,3);
                nv=nv(mon_avg_o:(len-(12-mon_avg_f)));
                % Reshape index so that each year has 12 monthly values
                nr=reshape(nv,12,length(nv)/12);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,nr(:,m_yrs));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 's'
                % SST
                olat=permute(ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/tos_Omon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','lat'),[2 1]);
                olon=permute(ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/tos_Omon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','lon'),[2 1]);
                ni=find(olon<0); olon(ni)=olon(ni)+360; % replace negative lon values
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/tos_Omon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','tos');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                Xv(Xv==0)=NaN; % replace nans
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'y'
                % Sea surface salinit(y)
                olat=permute(ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/sos_Omon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','lat'),[2 1]);
                olon=permute(ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/sos_Omon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','lon'),[2 1]);
                ni=find(olon<0); olon(ni)=olon(ni)+360; % replace negative lon values
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2g-pi/sos_Omon_GFDL-ESM2G_piControl_r1i1p1_000101-050012.nc','sos');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                Xv(Xv==0)=NaN; % replace nans
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'gfdlesm2m_pi')
    
    lat=ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/tas_Amon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','lat');
    lon=ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/tas_Amon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','lon');
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                % Surface temperature
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/tas_Amon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','tas');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'g'
                % Geopotential height
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/zg200_Amon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','zg'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'l'
                % Soil moisture
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/mrso_Lmon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','mrso');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                % Just use land area
                sftlf=permute(ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/sftlf_fx_GFDL-ESM2M_piControl_r0i0p0.nc','sftlf'),[2 1]);
                lndidx=find((sftlf(:)>0));
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                Xv=reshape(Xv(:,:,m_yrs),rows*cols,len);
                Xv=Xv(lndidx,:);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,Xv);
                id_X(2*j)=size(X,1);
            case 'z'
                % ITCZ location (can be seasonal or annual)
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/pr_Amon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','pr');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                ITCZ = itcz(Xv,lat,lon);
                len=length(ITCZ(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(ITCZ(m_yrs),1,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'i'
                % SPEI 12
                load('/d1/nsteiger/climate-data/gfdl-esm2m-pi/spei_output_scl_12_krnl_e_u2_gcm_03-Aug-2016_12:12:21.mat','spei_lndonly')
                %mon_o=12;mon_f=2; disp('DJF SPEI12')
                %mon_o=6;mon_f=8; disp('JJA SPEI12')
                %Xv=mon2ann(spei_lndonly,mon_o,mon_f);
                %if(mon_avg_o>mon_avg_f)&&(mon_f>mon_o);Xv=Xv(:,1:end-1);end
                Xv=mon2ann(spei_lndonly,mon_avg_o,mon_avg_f);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_X(2*j)=size(X,1);
                clear Xv
                
            case 'r'
                % PDSI (palme*r* drought severity index)
                % single uniform available water content
                load('/d1/nsteiger/climate-data/gfdl-esm2m-pi/pdsi_output_AWC_c_7_u2_gcm_03-Aug-2016_12:13:35.mat','pdsi_lndonly')
                %mon_o=12;mon_f=2; disp('DJF PDSI')
                %mon_o=6;mon_f=8; disp('JJA PDSI')
                %Xv=mon2ann(pdsi_lndonly,mon_o,mon_f);
                %if(mon_avg_o>mon_avg_f)&&(mon_f>mon_o);Xv=Xv(:,1:end-1);end
                Xv=mon2ann(pdsi_lndonly,mon_avg_o,mon_avg_f);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_X(2*j)=size(X,1);
                clear Xv
            case 'o'
                % AMO (annual, using skin temperature on regular lat/lon grid)
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/ts_Amon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','ts');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                load('/d1/nsteiger/climate-data/gfdl-esm2m-pi/north_atlantic_ocean_mask.mat')
                AMO = amo(Xv,atlntc_mask,lat,0);%0=no smoothing
                len=length(AMO(m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(AMO(m_yrs),1,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'n'
                % Nino3.4 (monthly, using skin temperature on regular lat/lon grid)
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/ts_Amon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','ts');
                Xv_mon=permute(Xv_mon,[2 1 3]);
                % compute nino index
                nv=nino(Xv_mon,lat,lon,3);
                % Account for different year definitions
                len=size(Xv_mon,3);
                nv=nv(mon_avg_o:(len-(12-mon_avg_f)));
                % Reshape index so that each year has 12 monthly values
                nr=reshape(nv,12,length(nv)/12);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,nr(:,m_yrs));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 's'
                % SST
                olat=permute(ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/tos_Omon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','lat'),[2 1]);
                olon=permute(ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/tos_Omon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','lon'),[2 1]);
                ni=find(olon<0); olon(ni)=olon(ni)+360; % replace negative lon values
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/tos_Omon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','tos');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                Xv(Xv==0)=NaN; % replace nans
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'y'
                % Sea surface salinit(y)
                olat=permute(ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/sos_Omon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','lat'),[2 1]);
                olon=permute(ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/sos_Omon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','lon'),[2 1]);
                ni=find(olon<0); olon(ni)=olon(ni)+360; % replace negative lon values
                Xv_mon=ncread('/d1/nsteiger/climate-data/gfdl-esm2m-pi/sos_Omon_GFDL-ESM2M_piControl_r1i1p1_000101-050012.nc','sos');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                Xv(Xv==0)=NaN; % replace nans
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
                
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'mpi_lm')
    
    load('/d1/nsteiger/climate-data/mpi_esm_last_mil/latlon_mpi.mat')
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                % Surface temperature
                load('/d1/nsteiger/climate-data/mpi_esm_last_mil/tas_mon_mpiesm_past1000.mat')
                Xv=mon2ann(tas_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear tas_mon
            case 'p'
                % Precipitation
                load('/d1/nsteiger/climate-data/mpi_esm_last_mil/pr_mon_past1000.mat')
                Xv=mon2ann(pr_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear pr_mon
            case 'l'
                % Soil moisture
                load('/d1/nsteiger/climate-data/mpi_esm_last_mil/mrso_r_mon_past1000.mat')
                Xv=mon2ann(mrso_r_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear mrso_r_mon
            case 'g'
                % zg 500
                zg500=squeeze(ncread('/d1/nsteiger/climate-data/mpi_esm_last_mil/zg_500hPa_Amon_MPI-ESM-P_past1000_085001-185012.nc','zg'));
                zg500=permute(zg500,[2 1 3]);
                Xv=mon2ann(zg500,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear zg500
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'echam5')
    
    load('/d1/nsteiger/climate-data/echam5/e5_latlon.mat')
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                % Surface temperature
                Xv_mon=ncread('/d1/nsteiger/climate-data/echam5/t2m_mon_1871_2011.nc','temp2');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=flip(permute(Xv,[2 1 3]),1);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'g'
                % Geopotential
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/echam5/zg500_mon_1871_2011.nc','geopoth'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=flip(permute(Xv,[2 1 3]),1);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'p'
                % Precipitation
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/echam5/precip_mon_1871_2011.nc','precip'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=flip(permute(Xv,[2 1 3]),1);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'z'
                % zonal precipitation
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/echam5/precip_mon_1871_2011.nc','precip'));
                Xv=squeeze(mean(mon2ann(Xv_mon,mon_avg_o,mon_avg_f),1)); % take time mean, then zonal mean
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'l'
                % Soil moisture
                load('/d1/nsteiger/climate-data/echam5/ws_landonly_mon_1871_2011.mat','ws_lndonly');
                %Xv=mon2ann(ws_lndonly,mon_avg_o,mon_avg_f);
                disp('Using JJA soil moisture...')
                Xv=mon2ann(ws_lndonly,6,8);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_X(2*j)=size(X,1);
                clear Xv
            case 'i'
                % SPEI 12
                load('/d1/nsteiger/climate-data/echam5/spei_output_scl_12_krnl_g_u2_gcm_25-May-2016_14:42:27.mat','spei_lndonly')
                disp('Loading *land-only* SPEI 12...')
                % Average data
                disp('Using JJA SPEI12...')
                Xv=mon2ann(spei_lndonly,6,8);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_X(2*j)=size(X,1);
                clear Xv
            case '3'
                % SPEI 12
                load('/d1/nsteiger/climate-data/echam5/spei_output_scl_3_krnl_g_u2_gcm_25-May-2016_15:37:40.mat','spei_lndonly')
                disp('Loading *land-only* SPEI 3...')
                % Average data
                disp('Using JJA SPEI3...')
                Xv=mon2ann(spei_lndonly,6,8);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_X(2*j)=size(X,1);
                clear Xv
                
            case 'r'
                % PDSI (palme*r* drought severity index)
                % single uniform available water content
                %load('/d1/nsteiger/climate-data/echam5/pdsi_output_AWC_7_u2_gcm_26-May-2016_09:47:53.mat')
                load('/d1/nsteiger/climate-data/echam5/pdsi_output_AWC_v_u2_gcm_26-May-2016_16:38:15.mat','pdsi_lndonly')
                disp('Loading *land-only* PDSI...')
                % Average data
                disp('Using JJA PDSI...')
                Xv=mon2ann(pdsi_lndonly,6,8);
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,Xv(:,m_yrs));
                id_X(2*j)=size(X,1);
                clear Xv
                
            case 'u'
                % U200 wind
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/echam5/u200_mon_1871_2011.nc','u'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=flip(permute(Xv,[2 1 3]),1);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
                
            case 'f'
                % stream function 200 hPa
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/echam5/streamfunc200_mon_1871_2011.nc','stream'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=flip(permute(Xv,[2 1 3]),1);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'e'
                % Precipitation minus Evaporation
                Xv_mon=double(ncread('/d1/nsteiger/climate-data/echam5/precip_minus_evap_mon_1871_2011.nc','pe'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=flip(permute(Xv,[2 1 3]),1);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'w'
                % surface wind
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/echam5/wind10_mon_1871_2011.nc','wind10'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=flip(permute(Xv,[2 1 3]),1);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 's'
                % sea level pressure
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/echam5/slp_mon_1871_2011.nc','slp'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=flip(permute(Xv,[2 1 3]),1);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'd'
                % d18Op
                Xv_mon=squeeze(ncread('/d1/nsteiger/climate-data/echam5/d18O_mon_1871_2011.nc','wisoaprt_d'));
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=flip(permute(Xv,[2 1 3]),1);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'echam5_2cam5iso')
    
    load('/d1/nsteiger/climate-data/cesm-iso/echam5_coarser_CAM5iso_latlon.mat')
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                % ECHAM COARSENED TO CAM5
                % tas
                load('/d1/nsteiger/climate-data/cesm-iso/echam5_coarser_CAM5iso_tas_mon.mat')
                Xv=mon2ann(tas_mon_crsr,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                
            case 's'
                % COARSENED VERSION OF ECHAM5 TO CAM5 RESOLUTION
                % sea level pressure
                load('/d1/nsteiger/climate-data/cesm-iso/echam5_coarser_CAM5iso_slp_mon.mat');
                Xv=mon2ann(slp_mon_crsr,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv
                
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'echam5_2speedy')
    
    load('/d1/nsteiger/climate-data/speedy/latlon_speedy.mat')
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                % ECHAM COARSENED TO SPEEDY
                % tas
                load('/d1/nsteiger/climate-data/echam5/tas_mon_coarsened2speedy.mat')
                Xv=mon2ann(tas_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                
            case 'g'
                % ECHAM COARSENED TO SPEEDY
                % geopotential 500 hPa
                load('/d1/nsteiger/climate-data/echam5/zg_mon_coarsened2speedy.mat');
                Xv=mon2ann(zg_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv
                
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'speedy_lm')
    
    load('/d1/nsteiger/climate-data/speedy/latlon_speedy.mat')
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                % Surface temperature
                load('/d1/nsteiger/climate-data/speedy/SPEEDY_Temp2m_R80_mon_1000_2005.mat');
                Xv_mon=Temp_2m;
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv_mon Xv
            case 'p'
                % Precipitation minus Evaporation [mm/day]
                load('/d1/nsteiger/climate-data/speedy/SPEEDY_Precip_R80_mon_1000_2005.mat');
                Xv_mon=Precip;
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv_mon Xv
            case 'g'
                % geopotential heights at 500 hPa
                load('/d1/nsteiger/climate-data/speedy/SPEEDY_GH500_R80_mon_1000_2005.mat');
                Xv_mon=GH_500;
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv_mon Xv
            otherwise
                error('Incorrect state specification.')
        end
    end
    
elseif strcmp(mdl_c,'icam5_hist')
    
    load('/d1/nsteiger/climate-data/icam/latlon_icam.mat')
    
    for j=1:length(state_tp)
        switch state_tp(j)
            case 't'
                % Surface temperature
                Xv_mon=ncread('/d1/nsteiger/climate-data/icam/tas_Amon_iCAM_historical_185001-201412.nc','TREFHT');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'p'
                % Precipitation minus Evaporation [m/s]
                Xv_mon=ncread('/d1/nsteiger/climate-data/icam/pr_Amon_iCAM_historical_185001-201412.nc','pr');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'g'
                % geopotential height at 500 hPa
                Xv_mon=ncread('/d1/nsteiger/climate-data/icam/zg-850-500-200_Amon_iCAM_historical_185001-201412.nc','Z');
                Xv_mon=squeeze(Xv_mon(:,:,2,:)); % get 500 hPa
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case '2'
                % geopotential height at 200 hPa
                Xv_mon=ncread('/d1/nsteiger/climate-data/icam/zg-850-500-200_Amon_iCAM_historical_185001-201412.nc','Z');
                Xv_mon=squeeze(Xv_mon(:,:,3,:)); % get 200 hPa
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            case 'd'
                % d18O of precip
                Xv_mon=ncread('/d1/nsteiger/climate-data/icam/d18O_pr_Amon_iCAM_historical_185001-201412.nc','d18O_pr');
                Xv=mon2ann(Xv_mon,mon_avg_o,mon_avg_f);
                Xv=permute(Xv,[2 1 3]);
                [rows,cols,len]=size(Xv(:,:,m_yrs));
                id_X(2*j-1)=size(X,1)+1;
                X=cat(1,X,reshape(Xv(:,:,m_yrs),rows*cols,len));
                id_X(2*j)=size(X,1);
                clear Xv Xv_mon
            otherwise
                error('Incorrect state specification.')
        end
    end
    
end


end

